package com.sanitysewer;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import javax.swing.*;

public class ViewingArea extends JComponent implements MouseListener,
                                                       MouseMotionListener {
    private int w, h;
    private BufferedImage bi;
    private Image image;
    private Graphics2D g2;

    // lights
    private Lights lights;
    private int lighting = Common.CONSTANT;

    // curves
    private Curve bc = null;

    // 3D models
    private WireFrame wireframe;
    private Projector projector = null;

    // flags and local data pertaining to transformations
    private boolean draggable = false;
    private int oldX , oldY;
    private Point dragP;

    private int mode = Common.CURVE2D;

    private double step = 10; // the steps between rotation in the wireframe
    private double angle = 360;

    private int spin = Common.YSPIN;

    private int curve = Common.BEZIER;

    public ViewingArea(int width, int height) {
	this.w = width;  this.h = height;
	this.setPreferredSize(new Dimension(w, h));

	this.addMouseListener(this);
	this.addMouseMotionListener(this);

	this.bi = new BufferedImage(w, h, BufferedImage.TYPE_3BYTE_BGR);
	this.g2 = (Graphics2D)bi.getGraphics();
	g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

	lights = new Lights();

	drawbg();
    }


    public int getWidth() { return w; }

    public void setMode(int i) { mode = i; }

    public void addLight(double intensity, double x, double y, double z, int type) {
	lights.addLight(intensity, x, y, z, type);
    }

    public void setLightingMode(int m) {
	if (projector == null) { lighting = m; }
	else {
	    lighting = m;
	    projector.setLightingMode(m);
	}
    }

    public void setSurface(double ka, double kd) { lights.setSurface(ka, kd); }

    public void setLight(double intensity, double x, double y, double z, int i) {
	lights.setLight(intensity, x, y, z, i);
    }
    
    public void setCurve(int c) { curve = c; }

    
    public void mouseClicked(MouseEvent me) { 
	int x = me.getX(); int y = me.getY();

	if (bc == null) {
	    
	    switch (curve) {
	    case Common.BEZIER:
		bc = new Bezier(g2);
		break;
	    case Common.BSPLINE:
		bc = new Bspline(g2);
		break;
	    }

	    bc.setStart(x, y);
	}
	else if (!bc.isEndSet()) {
	    bc.setEnd(x, y);
	}
	else if (!bc.isControl1Set()) {
	    bc.setControl1(x, y);
	}
	else if (!bc.isControl2Set()) {
	    bc.setControl2(x, y);
	}
	else if (bc instanceof Bspline) {
	    bc.setControlN(x, y);
	}

	bc.render();
	repaint();
	
    }

    public void mousePressed(MouseEvent me){ 
	int x = me.getX(); int y = me.getY();
	oldX = x;  oldY = y;

	if (bc != null && mode == Common.CURVE2D) {
	    if (bc.isControl2Set() &&
		(dragP = bc.getDragPoint(x, y)) != null) {
		draggable = true;
	    }
	}


    }


    public void mouseReleased(MouseEvent me){ 
	draggable = false;
    }

    public void mouseEntered(MouseEvent me){ }
    public void mouseExited(MouseEvent me){ }
    public void mouseDragged(MouseEvent me){ 
	int x = me.getX(); int y = me.getY();

	if (draggable && mode == Common.CURVE2D) {
	    drawbg();  // necessary since XOR mode leave trailers
	    bc.setDrag(dragP, x - oldX, y - oldY);
	    bc.render();
	    repaint();
	    oldX = x; oldY = y;
	}
	else if (mode == Common.WIREFRAME || mode == Common.POLYGONS) {
	    projector.rotate((double)(x - oldX), (double)(oldY - y));
	    oldY = y; oldX = x;
	    drawbg();
	    projector.render(mode);
	    repaint();
	}
	
    }
    public void mouseMoved(MouseEvent me){ }


    public void setSpinDirection(int s) { 
	spin = s; 

	if (mode == Common.WIREFRAME || mode == Common.POLYGONS) {	    
	    wireframe = new WireFrame(bc.getDoubleArray(), angle, step, spin);
	    projector.updateWireFrame(wireframe);
	    drawbg();
	    projector.render(mode);
	}

	repaint();
    }


    public boolean updateCurveIncrement(double inc) {
	
	if (bc == null) { return false; }
	else {
	    bc.setIncrement(inc);
	    drawbg();
	}


	if (mode == Common.CURVE2D) {
	    bc.render();
	}
	else if (mode == Common.WIREFRAME || mode == Common.POLYGONS) {
	    wireframe = new WireFrame(bc.getDoubleArray(), angle, step, spin);
	    projector.updateWireFrame(wireframe);
	    drawbg();
	    projector.render(mode);
	}

	repaint();

	return true;
    }


    public boolean updateRotationSteps(double s) {
	this.step = s;

	if (bc == null) { return false; }
	else if (mode == Common.WIREFRAME || mode == Common.POLYGONS) {
	    wireframe = new WireFrame(bc.getDoubleArray(), angle, step, spin);
	    projector.updateWireFrame(wireframe);
	    drawbg();
	    projector.render(mode);
	}	
	
	repaint();

	return true;
    }

    public boolean updateRotationAngle(double a) {
	this.angle = a;

	if (bc == null) { return false; }
	else if (mode == Common.WIREFRAME || mode == Common.POLYGONS) {
	    wireframe = new WireFrame(bc.getDoubleArray(), angle, step, spin);
	    projector.updateWireFrame(wireframe);
	    drawbg();
	    projector.render(mode);
	}	
	
	repaint();

	return true;
    }
    

    public void buildCurve() {
	
	if (bc == null) {
	    
	}
	else {
	    drawbg();
	    bc.render();
	    repaint();
	}

    }

    
    public void rerender() {

	if (wireframe == null || projector == null) {  }
	else {
	    drawbg();
	    projector.setLights(lights);
	    projector.render(mode);
	    repaint();
	}
    }


    public void buildSolidFrame() {  buildWireFrame();  }


    public void buildWireFrame() {
	
	if (bc.isControl2Set()) {

	    

	    wireframe = new WireFrame(bc.getDoubleArray(), angle, step, spin);
	    projector = new Projector(g2, wireframe, w, h, lighting);

	    drawbg();
	    projector.setLights(lights);
	    projector.render(mode);
	    repaint();
	}

    }

    public boolean canBuildModel() { 
	if (bc == null || !bc.isControl2Set()) { return false; }
	else { return true; }
    }

    public void clear() {
	mode = Common.CURVE2D;
	bc = null;
	wireframe = null;
	projector = null;
	drawbg();
	repaint();
    }

    private void drawbg() {

	// back ground field
	g2.setColor(Common.BGCOLOR);
	g2.fillRect(0, 0, w, h);

	// axis
	g2.setColor(Common.TICKCOLOR);
	g2.drawLine(Common.WIDTH/2, 0, Common.WIDTH/2, Common.HEIGHT);
	g2.drawLine(0, Common.HEIGHT/2, Common.WIDTH, Common.HEIGHT/2);
    }

    public void paintComponent(Graphics g){
	image = (Image)bi;
	g.drawImage(image, 0, 0, null);
    }
    


}
