/* TO DO'S
 *
 *
 * copyright University of Victoria, 2004
 */


#include "logo.h"
#include "globals.h"
#include "sound.h"
#include <stdlib.h>
#include <float.h>

int location = 0; /* index of current sample */

NODE *lsinewave(NODE *args) {
  NODE *f, *array;
  int frequency;

  f = numeric_arg(args);
  if (NOT_THROWING) {
    frequency = (nodetype(f) == FLOATT) ? (FIXNUM)getfloat(f) : getint(f);
  }

  int table_size = SAMPLE_RATE/frequency;
  array = make_array(table_size);
  setarrorg(array, 0);

  int i;
  for (i=0; i<getarrdim(array); i++) {
    float sample = (float) (AMPLITUDE * sin( (((double)i/(double)(table_size)) * 2 * M_PI )));
    (getarrptr(array))[i] = newnode(FLOATT);
    setfloat((getarrptr(array))[i], sample);
  }

  return(array);
}


NODE *ltrianglewave(NODE *args) {
  NODE *f, *array;
  int frequency;

  f = numeric_arg(args);
  if (NOT_THROWING) {
    frequency = (nodetype(f) == FLOATT) ? (FIXNUM)getfloat(f) : getint(f);
  }

  int table_size = SAMPLE_RATE/frequency;
  array = make_array(table_size);
  setarrorg(array, 0);

  double x = 0.0;
  double deltax = (2 * M_PI) / (double)table_size;

  int i;
  for (i=0; i<getarrdim(array); i++) {
    float sample = 0.0;

    /* The pos variable limits calcultion to one triangle which can then be duplicated
     * and inverted to complete the cycle. */
    double pos = x;
    if (pos > M_PI){ pos = x - M_PI; }

    if (pos <= M_PI/2){
      sample = (0.9/(M_PI/2)) * pos;
    }
    else {
      sample = (((-0.9/(M_PI/2))) * pos) + 1.8;
    }
    
    if (x > M_PI && x <= 2*M_PI) { sample*=-1; }


    (getarrptr(array))[i] = newnode(FLOATT);
    setfloat((getarrptr(array))[i], sample);

    x += deltax;
  }

  return(array);
}


NODE *lsquarewave(NODE *args) {
  NODE *f, *array;
  int frequency;

  f = numeric_arg(args);
  if (NOT_THROWING) {
    frequency = (nodetype(f) == FLOATT) ? (FIXNUM)getfloat(f) : getint(f);
  }

  int table_size = SAMPLE_RATE/frequency;
  array = make_array(table_size);
  setarrorg(array, 0);

  double x = 0.0;
  double deltax = (2 * M_PI) / (double)table_size;

  int i;
  float sample;
  for (i=0; i<getarrdim(array); i++) {
    if (x <= M_PI){ sample = 0.9; }
    else { sample = -0.9; }

    (getarrptr(array))[i] = newnode(FLOATT);
    setfloat((getarrptr(array))[i], sample);
    
    x += deltax;
  }  
  

  return(array);
}


NODE *lnoise(NODE *args) {
  NODE *oarray, *d;
  long duration;  /* number of milliseconds of noise to return */

  d = numeric_arg(args);
  if (NOT_THROWING){ duration = (nodetype(d) == FLOATT? (FIXNUM)getfloat(d): getint(d)); }
  else { err_logo(BAD_DATA, d); return UNBOUND; }

  long length = duration * SAMPLE_RATE/1000;
  oarray = make_array(length);
  setarrorg(oarray, 0);

  int i=0, r;
  float sample;
  for (; i<length; i++){
    r = rand(); 
    sample = ((float)r/(RAND_MAX/2)) - 1.0;
 
    (getarrptr(oarray))[i] = newnode(FLOATT);
    setfloat((getarrptr(oarray))[i], sample);   
  }

  return(oarray);
}

NODE *lcopywave(NODE *args) {
  NODE *new_array, *array, *times;
  int i, j, k, n, size;

  array = car(args);
  if (nodetype(array) != ARRAY) {
    err_logo(BAD_DATA, array);  return UNBOUND;
  }

  times = numeric_arg(cdr(args));
  if (NOT_THROWING) {
    n = (nodetype(times) == FLOATT) ? (FIXNUM)getfloat(times) : getint(times);
  }
  else {
    err_logo(BAD_DATA, n);  return UNBOUND;
  }
  if (n < 0) {
    err_logo(BAD_DATA, times);  return UNBOUND;
  }

  size = getarrdim(array) * n;
  new_array = make_array(size);
  setarrorg(new_array, 0);

  /* works, but too slowly... i can't think of any method for speeding it up
   * such as an array of pointers, some struct that maintains one copy of the
   * the cycle plus the number of iterations, etc... that doesn't necessitate
   * chaning the underlying logo data structure logo_node.  but, maybe i'm
   * missing something? */
  k=0;
  for (i=0; i<n; i++) {
    for (j=0; j<getarrdim(array); j++){
      (getarrptr(new_array))[k] = newnode(FLOATT);
      setfloat((getarrptr(new_array))[k], getfloat((getarrptr(array))[j]));
      k++;
    }
  }

  return(new_array);
}


NODE *lwaveenvelope(NODE *args) {
  NODE *array, *liSt, *pair, *a, *d;
  int i;

  array = car(args);
  liSt = (NODE *)list_arg(cdr(args));

  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }
  if (!is_list(liSt)){ err_logo(BAD_DATA, liSt); return UNBOUND; }

  float factor = 1.0;

  float currentmsec, sample;
  float oldx = 0.0;
  float oldy = 0.0;
  float thresholdmsec = 0.0;
  float thresholdfactor = 0.0;
  for (i=0; i<getarrdim(array); i++) {
    sample = getfloat((getarrptr(array))[i]);
    currentmsec = (float)i/((float)SAMPLE_RATE/1000.0);

    if (currentmsec >= thresholdmsec) { /* we have crossed a time threshold */
 
      oldx = thresholdmsec;
      oldy = thresholdfactor;

      if (NOT_THROWING && liSt != NIL) { /* check args and retrieve their values */
	pair = car(liSt);
	liSt = cdr(liSt);

	if (is_list(pair) && NOT_THROWING) {
	  a = numeric_arg(pair);
	  d = numeric_arg(cdr(pair));

	  if (NOT_THROWING) {
	    thresholdfactor = (nodetype(a) == FLOATT? getfloat(a): (FLONUM)getint(a));
	    thresholdmsec += (nodetype(d) == FLOATT? (FIXNUM)getfloat(d): getint(d));

	    while (thresholdfactor > 1) { thresholdfactor = thresholdfactor / 10; }
	  }
	  else { err_logo(BAD_DATA, pair); return UNBOUND; }
	}
	else { err_logo(BAD_DATA, pair); return UNBOUND; }
      }
    }

    float rise = thresholdfactor - oldy;
    float run  = thresholdmsec - oldx;
    float intersection = oldy - ((rise/run)*oldx);

    factor = (rise/run)*currentmsec + intersection;
    sample *= factor;
    setfloat((getarrptr(array))[i], sample);
  }

  return(array);
}


NODE *lcombinewaves(NODE *args) {
  NODE *liSt, *farray, *carray;
  int longest = 0;

  liSt = car(args);
  if (!is_list(liSt) || !NOT_THROWING){ err_logo(BAD_DATA, args); return UNBOUND; }

  /* determine which array is longest */
  while (liSt != NIL && NOT_THROWING) {
    carray = car(liSt); liSt = cdr(liSt);

    if (nodetype(carray) != ARRAY){ err_logo(BAD_DATA, carray); return UNBOUND; }
    if (nodetype((getarrptr(carray))[0]) != FLOATT){ err_logo(BAD_DATA, carray); return UNBOUND; }

    if (longest < getarrdim(carray)){ longest = getarrdim(carray); }
  }

  /* create the new array which has the length of the longest argument array */
  farray = make_array(longest);
  setarrorg(farray, 0);

  /* now copy the arrays in such that they fill the whole new array...
   * note that this could mean some waves appear to end very abruptly, ie
   * not at a near-zero value.  one could take care of this using the 
   * waveenvelope function. */
  liSt = car(args);
  int i=0;
  for (; i<longest; i++){
    float sample=0;

    while (liSt != NIL){
      carray = car(liSt); liSt = cdr(liSt);
 
      int j = i;
      while (j >= getarrdim(carray)){ j = j - getarrdim(carray);}

      sample += getfloat((getarrptr(carray))[j]);
    }
    liSt = car(args); /* resest liSt */

    (getarrptr(farray))[i] = newnode(FLOATT);
    setfloat((getarrptr(farray))[i], sample);
  }


  return(farray);
}


NODE *laddwaveat(NODE *args) {
  NODE *o, *farray, *tarray, *array;
  unsigned long offset;

  tarray = car(args); args = cdr(args);    /* target array */
  array  = car(args); args = cdr(args);    /* array to add */
  if (nodetype(tarray) != ARRAY){ err_logo(BAD_DATA, tarray); return UNBOUND; }
  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }


  o = numeric_arg(args);
  if (NOT_THROWING){
    offset = (nodetype(o) == FLOATT? (FIXNUM)getfloat(o): getint(o));
  }
  else { err_logo(BAD_DATA, o); return UNBOUND; }

  offset *= 44.1;  /* convert offset from msec to samples */
  
  /* create the final array */
  unsigned long length = 0;
  if (getarrdim(tarray) > offset + getarrdim(array)){ length = getarrdim(tarray); }
  else { length = offset + getarrdim(array); }
  farray = make_array(length);
  setarrorg(farray, 0);

  unsigned long i=0,j=0;
  for (; i<getarrdim(farray); i++){
    float sample = 0.0;

    if (i < getarrdim(tarray)){ sample += getfloat((getarrptr(tarray))[i]); }
    if (i >= offset && j < getarrdim(array)){ sample += getfloat((getarrptr(array))[j]); j++;}

    (getarrptr(farray))[i] = newnode(FLOATT);
    setfloat((getarrptr(farray))[i], sample);
  
  }
  

  return(farray);
}


NODE *lplaywave(NODE *args) {
  PaError err;
  PortAudioStream *stream;

  if (SOUNDON != 1) { return(UNBOUND); }                /* ensure portaudio has been initialized */
  if (nodetype(car(args)) != ARRAY ||
      nodetype((getarrptr(car(args)))[0]) != FLOATT) {  /* check the arguments right away */
    err_logo(BAD_DATA, args);
    return(UNBOUND);
  }

  err = Pa_OpenStream(
		      &stream,
		      paNoDevice,     /* default input device */
		      0,              /* no input */
		      paFloat32,      /* 32 bit floating point input */
		      NULL,
		      Pa_GetDefaultOutputDeviceID(),
		      2,              /* stereo output */
		      paFloat32,      /* 32 bit floating point output */
		      NULL,
		      SAMPLE_RATE,
		      FRAMES_PER_BUFFER,
		      0,              /* number of buffers, if zero then use default minimum */
		      paClipOff,      /* clipping accounted for elsewhere */
		      playwaveCallback,
		      args );
  if (err != paNoError){ error(); }
  err = Pa_StartStream( stream );
  if( err != paNoError ) { error(); }
  
  while (location < getarrdim(car(args))) { /* poll until done */
    Pa_Sleep( 10 );
  }
  
  err = Pa_StopStream( stream );
  if( err != paNoError ) { error(); }
  err = Pa_CloseStream( stream );
  if( err != paNoError ) { error(); }
  
  /* reset globals */
  location = 0;

  return(UNBOUND);
}

NODE *lrecordwave(NODE *args){ 
  PaError err;
  PortAudioStream *stream;
  NODE *oarray, *n;
  int time;

  if (SOUNDON != 1) { return(UNBOUND); } /* ensure portaudio has been initialized */

  n = numeric_arg(args);

  if (NOT_THROWING) {
    time = (nodetype(n) == FLOATT) ? (FIXNUM)getfloat(n) : getint(n);
  }
  else { err_logo(BAD_DATA, n); return UNBOUND; }

  if (time < 0) { err_logo(BAD_DATA, n); }

  oarray = make_array( time * SAMPLE_RATE/1000 );  /* will hold the recorded floats*/
  setarrorg(oarray, 0);

  /* this is a hack which I haven't yet been able to remove. */
  /* for some reason i need to set these nodes before passing this */
  /* array to the callback below.  must be something simple, but I */
  /* don't see it!!! arghhhh.... */
  unsigned long i = 0;
  for (; i<getarrdim(oarray); i++) {
    getarrptr(oarray)[i] = newnode( FLOATT );
  }

  err = Pa_OpenStream(
		      &stream,
		      Pa_GetDefaultInputDeviceID(),     /* default input device */
		      1,
		      paFloat32,      /* 32 bit floating point input */
		      NULL,
		      paNoDevice,
		      0,              /* stereo output */
		      paFloat32,      /* 32 bit floating point output */
		      NULL,
		      SAMPLE_RATE,
		      FRAMES_PER_BUFFER,
		      0,              /* number of buffers, if zero then use default minimum */
		      paClipOff,      /* clipping accounted for elsewhere */
		      recordwaveCallback,
		      oarray );

  location = 0; /* just in case it has't been reset somewhere */

  if (err != paNoError){ error(); }
  err = Pa_StartStream( stream );
  if( err != paNoError ) { error(); }

  printf("Start recording\n");

  Pa_Sleep( time );
/*   while (location < getarrdim(oarray)) { */
/*     Pa_Sleep( 10 ); */
/*   } */

  printf("Stop recording\n");

  err = Pa_StopStream( stream );
  if( err != paNoError ) { error(); }
  err = Pa_CloseStream( stream );
  if( err != paNoError ) { error(); }

  location = 0; /* reset this global */

  return oarray; 
}


NODE *lwavetable(NODE *args){
  NODE *liSt;
  NODE *b, *e;
  int begin, end;

  liSt = (NODE *)list_arg(cdr(args));
  if (!is_list(liSt)){ err_logo(BAD_DATA, liSt); return UNBOUND; }

  b = numeric_arg(liSt); liSt = cdr(liSt);
  if (NOT_THROWING){ begin = (nodetype(b) == FLOATT? (FIXNUM)getfloat(b): getint(b)); }
  else { err_logo(BAD_DATA, b); return UNBOUND; }

  e = numeric_arg(liSt);
  if (NOT_THROWING){ end = (nodetype(e) == FLOATT? (FIXNUM)getfloat(e): getint(e)); }
  else { err_logo(BAD_DATA, e); return UNBOUND; }

  /* limit the cut to sample rate */
  if (end - begin > SAMPLE_RATE/10 ||
      end - begin < -1*SAMPLE_RATE/10) {
    printf("Sorry, too big a clip.  Try an interval smaller than %d.\n", SAMPLE_RATE/10);
    return UNBOUND;
  }



  return lcutwave(args);
}

NODE *lcutwave(NODE *args){ 
  NODE *iarray, *oarray; /* input array, output array */
  NODE *liSt;            /* two item segment list [start stop] */
  NODE *b, *e;
  int begin, end, length;

  iarray = car(args);
  if (nodetype(iarray) != ARRAY){ err_logo(BAD_DATA, iarray); return UNBOUND; }

  liSt = (NODE *)list_arg(cdr(args));
  if (!is_list(liSt)){ err_logo(BAD_DATA, liSt); return UNBOUND; }

  b = numeric_arg(liSt); liSt = cdr(liSt);
  if (NOT_THROWING){ begin = (nodetype(b) == FLOATT? (FIXNUM)getfloat(b): getint(b)); }
  else { err_logo(BAD_DATA, b); return UNBOUND; }

  e = numeric_arg(liSt);
  if (NOT_THROWING){ end = (nodetype(e) == FLOATT? (FIXNUM)getfloat(e): getint(e)); }
  else { err_logo(BAD_DATA, e); return UNBOUND; }

  length = end - begin;
  if (length < 0){ length *= -1; }

  oarray = make_array(length);
  setarrorg(oarray, 0);
  
  long unsigned i=0, j=begin;

  for (; i<length; i++, j++){
    float sample = 0.0;

    if (j<getarrdim(iarray)){
      sample = getfloat((getarrptr(iarray))[j]);
    }

    (getarrptr(oarray))[i] = newnode(FLOATT);
    setfloat((getarrptr(oarray))[i], sample); 
  }

  return oarray; 
}


NODE *lreversewave(NODE *args) {
  NODE *iarray, *oarray;

  iarray = car(args);
  if (nodetype(iarray) != ARRAY){ err_logo(BAD_DATA, iarray); return UNBOUND; }

  oarray = make_array( getarrdim(iarray) );
  setarrorg(oarray, 0);  

  long unsigned i, j;
  for (i=0, j=getarrdim(iarray)-1; i<getarrdim(iarray); i++, j--) {
    (getarrptr(oarray))[i] = newnode(FLOATT);
    setfloat((getarrptr(oarray))[i], getfloat((getarrptr(iarray))[j]));
  }

  return oarray;
}


int recordwaveCallback( void *inputBuffer, void *outputBuffer,
			unsigned long framesPerBuffer,
			PaTimestamp outTime, void *userData ) {


  float *input = (float *)inputBuffer;
  NODE *buffer = (NODE *)userData;
  unsigned long i = 0;

  unsigned long framesLeft;

  if (getarrdim(buffer) - location < framesPerBuffer) {
    framesLeft = getarrdim(buffer) - location;
  }
  else {
    framesLeft = framesPerBuffer;
  }

  for (; i<framesLeft; i++) {
    setfloat(getarrptr(buffer)[location], *input++);
    location += 1;
  }


  return 0;
}

int playwaveCallback( void *inputBuffer, void *outputBuffer,
		      unsigned long framesPerBuffer,
		      PaTimestamp outTime, void *userData ) {
  float *out = (float *)outputBuffer;
  NODE *args = (NODE *)userData;
  NODE *array = car(args); /* type should already have been checked */
  unsigned long i = 0;
  
  for (; i<framesPerBuffer; i++) {
    float sample;
    if (location < getarrdim(array)){ 
      /* this is necessary in case we have some empty indices */
      if (nodetype((getarrptr(car(args)))[location]) != FLOATT) {
	sample = 0.0;
      }
      else {
	sample = (float)getfloat((getarrptr(array))[location]); 
      }
    }
    else { sample = 0.0; }

    *out++ = sample; /* left */
    *out++ = sample; /* right */

    location++;
  }

  
  return( 0 );
}



