/******* TO DO'S ***********************************************/
/* copyright University of Victoria, 2004 */

#include "sound.h"

/* some global keepers of state */
int count = 1;
float count_down = (float)SAMPLE_RATE/100.0;

int SOUNDON = 1;

NODE *lsoundon(NODE *args){
  PaError err;

  err = Pa_Initialize();
  if( err != paNoError ) { error(); }
  SOUNDON = 1;

  return(UNBOUND);
}


NODE *lsoundoff(NODE *args){

  Pa_Terminate();
  SOUNDON = -1;

  return(UNBOUND);
}


NODE *lrest(NODE *args){
  NODE *p;
  FIXNUM duration;

  p = numeric_arg(args);
  duration = (nodetype(p) == FLOATT) ? (FIXNUM)getfloat(p) : getint(p);

  Pa_Sleep( duration );
  
  return(UNBOUND);
}


NODE *ltime(NODE *args) {
  NODE *t;
  t = newnode(FLOATT);
  setfloat(t, -1.0);

#ifdef unix
  struct timeval tval;
  gettimeofday(&tval, NULL);
  setfloat(t, (tval.tv_sec)*1000.0 + (tval.tv_usec)/1000.0);
#endif
  

  return t;

}

NODE *ltone(NODE *args) {
  OscillatorData data;
  
  data.type = TONET;
  
  NODE *p, *d, *a;
  FIXNUM pitch, duration;
  FLONUM amplitude;
  
  p = numeric_arg(args);  if (stopping_flag == THROWING) return UNBOUND;
  a = numeric_arg(cdr(args));  if (stopping_flag == THROWING) return UNBOUND;
  d = numeric_arg(cddr(args));  if (stopping_flag == THROWING) return UNBOUND;
  
  /* check arguments */
  if (NOT_THROWING) {
    pitch = (nodetype(p) == FLOATT) ? (FIXNUM)getfloat(p) : getint(p);
    duration = (nodetype(d) == FLOATT) ? (FIXNUM)getfloat(d) : getint(d);
    amplitude = (nodetype(a) == FLOATT) ? (FLONUM)getfloat(a) : getint(a);
    if (amplitude > .9){ amplitude = .9; }
    if (pitch < 0 || duration < 0) {
      // do nothing
      return(UNBOUND);
    }
  }
  
  
  /* initialize sinusoidal wavetable */
  data.table_size = SAMPLE_RATE/pitch;
  float table[data.table_size];
  data.sine = table;
  makeWaveTable(pitch, amplitude, &data);
  
  data.sustain.duration = (double)duration;
  data.left_phase = data.right_phase = 0;
  
  playtone(&data, duration);
  
  return(UNBOUND);
}


NODE *ltonewt(NODE *args) {
  OscillatorData data;
  NODE *array, *liSt;
  NODE *p, *d, *a;
  FIXNUM pitch, duration;
  FLONUM amplitude;
  data.type = TONET;



  array = car(args);
  liSt = (NODE *)list_arg(cdr(args));
  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }
  if (!is_list(liSt)){ err_logo(BAD_DATA, liSt); return UNBOUND; }

  p = numeric_arg(liSt);  if (stopping_flag == THROWING) return UNBOUND;
  a = numeric_arg(cdr(liSt));  if (stopping_flag == THROWING) return UNBOUND;
  d = numeric_arg(cddr(liSt));  if (stopping_flag == THROWING) return UNBOUND;

  /* check arguments */
  if (NOT_THROWING) {
    pitch = (nodetype(p) == FLOATT) ? (FIXNUM)getfloat(p) : getint(p);
    duration = (nodetype(d) == FLOATT) ? (FIXNUM)getfloat(d) : getint(d);
    amplitude = (nodetype(a) == FLOATT) ? (FLONUM)getfloat(a) : getint(a);
    if (amplitude > .9){ amplitude = .9; }
    if (pitch < 0 || duration < 0) { return(UNBOUND); }
  }  

 
  /* initialize wavetable based on provided array */
  data.table_size = SAMPLE_RATE/pitch;
  float table[data.table_size];
  data.sine = table;  /* use the sine element so this will work with soundCallBack */
  copyWaveTable(pitch, amplitude, array, &data);
  data.sustain.duration = (double)duration;
  data.left_phase = data.right_phase = 0;

  playtone(&data, duration);

  return(UNBOUND);
}


void playtone(OscillatorData *data, FIXNUM duration) {
    PortAudioStream *stream;
    PaError err;

    if (SOUNDON == 1) {
      err = Pa_Initialize();
      if( err != paNoError ) { error(); }
      err = Pa_OpenStream(
			  &stream,
			  paNoDevice,     /* default input device */
			  0,              /* no input */
			  paFloat32,      /* 32 bit floating point input */
			  NULL,
			  Pa_GetDefaultOutputDeviceID(),
			  2,              /* stereo output */
			  paFloat32,      /* 32 bit floating point output */
			  NULL,
			  SAMPLE_RATE,
			  FRAMES_PER_BUFFER,
			  0,              /* number of buffers, if zero then use default minimum */
			  paClipOff,      /* clipping is accounted for elsewhere */
			  soundCallback,
			  data );
      if (err != paNoError){ error(); }
      err = Pa_StartStream( stream );
      if( err != paNoError ) { error(); }
      
      Pa_Sleep( duration );
      
      err = Pa_StopStream( stream );
      if( err != paNoError ) { error(); }
      err = Pa_CloseStream( stream );
      if( err != paNoError ) { error(); }
      Pa_Terminate();

      /* yikes, resetting globals */
      reset_sound_globals();
    }


}


void playharmony(OscillatorData *first_osc, FIXNUM duration) {
  PaError err;
  PortAudioStream *stream;


  /* okay, if the sound's on, initiate the callback via portaudio */
  if (SOUNDON == 1){
    err = Pa_OpenStream(
			&stream,
			paNoDevice,     /* default input device */
			0,              /* no input */
			paFloat32,      /* 32 bit floating point input */
			NULL,
			Pa_GetDefaultOutputDeviceID(),
			2,              /* stereo output */
			paFloat32,      /* 32 bit floating point output */
			NULL,
			SAMPLE_RATE,
			FRAMES_PER_BUFFER,
			0,              /* number of buffers, if zero then use default minimum */
			paClipOff,      /* clipping accounted for elsewhere */
			harmonyCallback,
			first_osc );
    if (err != paNoError){ error(); }
    err = Pa_StartStream( stream );
    if( err != paNoError ) { error(); }

    Pa_Sleep( duration );
    
    err = Pa_StopStream( stream );
    if( err != paNoError ) { error(); }
    err = Pa_CloseStream( stream );
    if( err != paNoError ) { error(); }
  }
  
  /* reset globals */
  reset_sound_globals();
}


NODE *lsound(NODE *args) {

  /* wrap the arguments in the appropriate list */
  args = cons(args, NIL);
  args = cons(args, NIL);

  return(lharmony(args));
}


NODE *lharmony(NODE *args) {
  OscillatorData *first_osc = 0;   /* start with a null for conditional below */
  int *d;
  *d = 0;

  first_osc = build_harmony(args, first_osc, d, NULL, SINUSOID);
  if ( first_osc == NULL ) { return NIL; }

  playharmony(first_osc, *d);
  free_oscillators(first_osc);
  
  return(UNBOUND);
}


/* wish I could figure out how to wrap the arguments and just reuse harmonywt */
/* another day... */
NODE *lsoundwt(NODE *args) {
  OscillatorData *first_osc = 0;   /* start with a null for conditional below */

  int *d;  
  *d = 0;

  NODE *array;


  array = car(args);
  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }

  first_osc = build_harmony(cons(cdr(args), NIL), first_osc, d, array, WAVEWT);
  if ( first_osc == NULL ) { return NIL; }

  playharmony(first_osc, *d);
  free_oscillators(first_osc);
  
  return UNBOUND;
}



NODE *lharmonywt(NODE *args) {
  OscillatorData *first_osc = 0;   /* start with a null for conditional below */

  int *d;  
  *d = 0;

  NODE *array;


  array = car(args);
  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }

  first_osc = build_harmony(cdr(args), first_osc, d, array, WAVEWT);
  if ( first_osc == NULL ) { return NIL; }

  playharmony(first_osc, *d);
  free_oscillators(first_osc);
  
  return UNBOUND;
}


OscillatorData *build_harmony(NODE *args, OscillatorData *first_osc, int *longest_duration, 
			      NODE *array, int type) {
  NODE *oscillator_list;

  oscillator_list = (NODE *)list_arg(args);
  if (stopping_flag == THROWING) return NULL;
  if (oscillator_list == NIL) return NULL;


  OscillatorData *prev_osc;
  while (oscillator_list != NIL  && NOT_THROWING) { /* loop through and extract each sound */
    NODE *current_wave;
    OscillatorData *data = oscalloc();

    current_wave = car(oscillator_list);
    oscillator_list = cdr(oscillator_list);
   
    if (is_list(current_wave)) { 
      NODE *p, *pairs;
      FIXNUM pitch;

      p = numeric_arg(current_wave);
      pairs = (NODE *)list_arg(cdr(current_wave));


      if (pairs == NIL) return NULL;
      if (stopping_flag == THROWING) return NULL;

      pitch = (nodetype(p) == FLOATT) ? (FIXNUM)getfloat(p) : getint(p);


 
      /* initialize sinusoidal wavetable */
      data->type = HARMONYT;
      data->table_size = SAMPLE_RATE/pitch;
      data->sine = (float *)malloc(sizeof(float) * data->table_size);


      if (type == SINUSOID) { makeWaveTable(pitch, AMPLITUDE, data); }
      else { copyWaveTable(pitch, AMPLITUDE, array, data); }

      data->left_phase = data->right_phase = 0;

      /* initialize the envelope data structure */
      EnvelopeData *firstenv;
      firstenv = getEnvelopeDataLinkedList(pairs, firstenv);
      if (firstenv == NULL) return NULL; /* make sure something's in the list */
      (EnvelopeData *)data->sustain.envelope = firstenv;
      (EnvelopeData *)data->current_envelope_segment = firstenv;

      /* create the linked list */
      if (first_osc == NULL) {
	first_osc = data;
	prev_osc  = data;
      }
      else {
	(OscillatorData *)prev_osc->next_oscillator = data;
	prev_osc = data;
      }

      /* check how long this "note" will run */
      int duration = getDurationFromLinkedList((EnvelopeData *)data->sustain.envelope);
      if (duration > *longest_duration) *longest_duration = duration;


    }
    else {
      err_logo(BAD_DATA, current_wave);
      return(NULL);
    }
  }
  prev_osc->next_oscillator = 0; /* for recognize the end of the list */

  return first_osc;
}

void free_oscillators(OscillatorData *first_osc) {

  /* free memory */
  OscillatorData *current = first_osc;
  while (current != NULL) {

    /* free linked list of envelope data */
    EnvelopeData *current_env = (EnvelopeData *)current->sustain.envelope;
    while (current_env != NULL) {
      free(current_env);
      current_env = (EnvelopeData *)current_env->next_envlop;
    }

    /* free the wave table */
    free(current->sine);

    /* free the oscillator data itself */
    OscillatorData *next = (OscillatorData *)current->next_oscillator;
    free(current);
    current = next;
  }

}


void reset_sound_globals(void) {
  count = 1;
  count_down = (float)SAMPLE_RATE/100.0;
}


int getDurationFromLinkedList(EnvelopeData *en) {
  EnvelopeData *current = en;
  EnvelopeData *next;
  int duration = 0;

  while (current != NULL) {
    duration += current->stop;
    current = (EnvelopeData *)current->next_envlop;
  }

  return(duration);
}

EnvelopeData *getEnvelopeDataLinkedList(NODE *pairs, EnvelopeData *initial_env) {

  int start = 0;
  EnvelopeData *previous_env;
  while (pairs != NIL && NOT_THROWING){
    NODE *pair;

    EnvelopeData *en = envalloc();
    
    pair  = car(pairs);
    pairs = cdr(pairs);

    if (pair == NIL || stopping_flag == THROWING) return (initial_env = 0);
    if (!is_list(pair)){
      err_logo(BAD_DATA, pair);
      return (initial_env = 0);
    }

    NODE *a, *d;
    FIXNUM duration;
    FLONUM amplitude;
      
    a = numeric_arg(pair); /* sure wish car and cdr worked like they do in lisp! */
    d = numeric_arg(cdr(pair));
    
    if ((nodetype(a) != FLOATT && nodetype(a) != INT) ||
	(nodetype(d) != INT && nodetype(d) != FLOATT)) {
      err_logo(BAD_DATA, pair);
      return (initial_env = 0);
    }
      
    if (NOT_THROWING) {
      amplitude = (nodetype(a) == FLOATT) ? (FLONUM)getfloat(a) : (FIXNUM)getint(a);
      duration  = (nodetype(d) == FLOATT) ? (FIXNUM)getfloat(d) : getint(d);

      if (amplitude > .9){ amplitude = .9; } /* avoid clipping */
      
      en->start = start;
      en->stop = start + duration;
      en->amplitude = amplitude;

      if (start != 0){
	(EnvelopeData *)previous_env->next_envlop = en;
      }
      else {
	initial_env = en;
      }
      previous_env = en;
      start = en->stop;
    }
  }
  previous_env->next_envlop = 0; /* marks end of list */

  return (initial_env);
}

EnvelopeData *envalloc(void) {
  return (EnvelopeData *)malloc(sizeof(EnvelopeData));
}


OscillatorData *oscalloc(void) {
  return (OscillatorData *)malloc(sizeof(OscillatorData));
}


void copyWaveTable(FIXNUM pitch, FLONUM amplitude, NODE *array, OscillatorData *data) {
  long unsigned i;
  float sample = 0.0;

  for ( i=0; i<data->table_size; i++  ) {
    long unsigned k = (long unsigned)(i*getarrdim(array)/data->table_size);
    data->sine[i] = amplitude * getfloat((getarrptr(array))[k]); 
    
  }

}


void makeWaveTable(FIXNUM pitch, FLONUM amplitude, void *userData) {
  OscillatorData *data = (OscillatorData*)userData;

  long unsigned i;
  for ( i=0; i<data->table_size; i++ ) {
    data->sine[i] = (float) (amplitude * sin( (((double)i/(double)(data->table_size)) * 2 * M_PI ))); 
  }

}


float getEnvelopeFactor(void *userData) {
  OscillatorData *data = (OscillatorData *)userData;
  float factor = 1.0;  /* default */
  EnvelopeData *current_envelope_segment;

  if (data->type == HARMONYT){
    float millisec = (float)count * (1000.0/(float)SAMPLE_RATE);
    current_envelope_segment = (EnvelopeData *)data->current_envelope_segment; /* just a synonym */

    /* initialize on first entrance to callback */
    if (count == 1){
      data->old_stop = 0;
      data->old_amplitude = 0.0;
    }

    /* update states if the next point defining envelope has been reached */
    if (millisec > (float)current_envelope_segment->stop &&
	current_envelope_segment->next_envlop != NULL){
      data->old_amplitude = current_envelope_segment->amplitude;
      data->old_stop = current_envelope_segment->stop;
      data->current_envelope_segment = current_envelope_segment->next_envlop;
      current_envelope_segment = (EnvelopeData *)data->current_envelope_segment;
    }
    

    float rise = current_envelope_segment->amplitude - data->old_amplitude;
    float run  = (float)current_envelope_segment->stop - (float)data->old_stop;
    float slope = rise / run;
    float intersect = data->old_amplitude - (slope * (float)data->old_stop);
    factor = slope * millisec + intersect;
    
    /* the callback's timing is a little less than perfect... this avoids overruns */
    if (millisec > current_envelope_segment->stop){ factor = 0.0; }
  }
  else if (data->type == TONET) {
    factor = 1.0;

    /* create a bit of an envlope to help dampen clicking at start and end, works okay */
    if (count < (int)((float)SAMPLE_RATE/100.0)){
      factor = pow((float)count/((float)SAMPLE_RATE/100.0), 2);
    }
    else if (((float)data->sustain.duration - (float)count/((float)SAMPLE_RATE/1000.0)) <= 10 &&
	     ((float)data->sustain.duration - (float)count/((float)SAMPLE_RATE/1000.0)) >= 0) {
      factor = pow((float)(count_down/((float)SAMPLE_RATE/100.0)), 2);
      count_down--;
    }
    else if (((float)data->sustain.duration - (float)count/((float)SAMPLE_RATE/1000.0)) < 0){
      factor = 0.0;
    }
  }
  else {
    factor = 1.0;
  }

  /* printf("%f\n", factor); */
  return(factor); 
}

float getSampleValueAt(int stereo, OscillatorData *data){
  float sample = 0.0;
  int n = 0;
  float factor = 0.0;
  OscillatorData *current = data;

  while (current != NULL) {

    factor = getEnvelopeFactor(current);

    if (stereo == STEREO_RIGHT) {
      sample += factor * current->sine[current->right_phase];
      current->right_phase += 1;
      if( current->right_phase >= current->table_size ) current->right_phase -= current->table_size;
    }
    else if (stereo == STEREO_LEFT) {
      sample += factor * current->sine[current->left_phase];
      current->left_phase += 1;
      if( current->left_phase >= current->table_size ) current->left_phase -= current->table_size;
    }

    n += 1;
    current = (OscillatorData *)current->next_oscillator;
  }
  
  /* normalize */
  if (sample != 0.0) sample = sample/(float)n;

  return(sample);
}

int harmonyCallback( void *inputBuffer, void *outputBuffer,
		     unsigned long framesPerBuffer,
		     PaTimestamp outTime, void *userData ) {
  OscillatorData *data = (OscillatorData*)userData;
  float *out = (float*)outputBuffer;
  unsigned long i;
  (void) outTime; /* Prevent unused variable warnings. */
  (void) inputBuffer;

  float factor = 1.0; /* default */

  for( i=0; i<framesPerBuffer; i++ ) {

    *out++ = getSampleValueAt(STEREO_LEFT, data);  /* stereo left */
    *out++ = getSampleValueAt(STEREO_RIGHT, data); /* stereo right */
    /* printf("%f\n", *out); */

    count++; /* used for ramping up/down amplitude */
  }

  return(0);
}

/* yup, this is just slightly modified from the portaudio pa_test_sine code.  i've given it
 * an envelope and removed the stereo offset. */
int soundCallback(  void *inputBuffer, void *outputBuffer,
		    unsigned long framesPerBuffer,
		    PaTimestamp outTime, void *userData ) {
  OscillatorData *data = (OscillatorData*)userData;
  float *out = (float*)outputBuffer;
  unsigned long i;
  (void) outTime; /* Prevent unused variable warnings. */
  (void) inputBuffer;
  
  float factor = 1.0;  /* default */


  for( i=0; i<framesPerBuffer; i++ ) {

    factor = getEnvelopeFactor(data);

    *out++ = (float)factor * data->sine[data->left_phase];  /* left */
    *out++ = (float)factor * data->sine[data->right_phase]; /* right */
    data->left_phase += 1;
    if( data->left_phase >= data->table_size ) data->left_phase -= data->table_size;
    data->right_phase += 1; /* we don't care about stereo right now */
    if( data->right_phase >= data->table_size ) data->right_phase -= data->table_size;
    
    count++; /* used for ramping up/down amplitude */
  }

  return(0);
}


void error(void){
  printf("There was an error with portaudio.\n");
}
