/* huffman.c
 * 
 * An implementation of huffman compression/decompression algorithm intended,
 * primarily, for use with the audio api.
 *
 * Aaron Hechmer, copyright University of Victoria, BC, 2005
 */

#include <stdlib.h>
#include "logo.h"
#include "sound.h"


HuffFreq *getFreqCounts( unsigned char *data, long length ){
  HuffFreq *hf = (HuffFreq *)malloc(sizeof(HuffFreq));
  long i=0;
  
  hf->keys   = (char *)malloc(sizeof(char) * 255);
  hf->values = (long *)malloc(sizeof(long) * 255);
  hf->active_size = 0;

  for (; i<length; i++) {
    if ( hf->values[data[i]] == NULL ) {
      hf->active_size += 1;
      hf->keys[data[i]] = data[i];
      hf->values[data[i]] = 1;
    }
    else { 
      hf->values[data[i]] += 1; 
    }  
  }
  
  /*  for (i=0; i<256; i++){ printf("%i\t%i\n", i, hf->values[i]); } */

  return(hf);

}


HuffFreq *sortHuffFreq( HuffFreq *hf_in ) {
  HuffFreq *hf_out = (HuffFreq *)malloc(sizeof(HuffFreq));
  
  hf_out->keys   = (char *)malloc(sizeof(char) * 255);
  hf_out->values = (long *)malloc(sizeof(long) * 255);

  hf_out = qsortHuffFreq( hf_in );


  freeHuff( hf_in );
  return(hf_out);

}

HuffFreq *qsortHuffFreq( HuffFreq *hf ) {

  if (hf->active_size <= 1){ return hf; } /* nothing left to sort here */

  HuffFreq *hf_left = hfalloc(1);
  HuffFreq *hf_right = hfalloc(1);
  HuffFreq *hf_pivot = hfalloc(1);
  
  hf_left->active_size = 0;
  hf_right->active_size = 0;
  hf_pivot->active_size = -1;
  
  long i=0, pivot;
  for (; i<hf->active_size; i++) {
    
    if (hf->values[i] == 0){ continue; }
    
    if (hf_pivot->active_size == -1) {
      hf_pivot->active_size = 0;
      pivot = hf->values[i];
    }

    if (hf->values[i] < pivot) { increment( hf_left, hf, i ); }
    else if (hf->values[i] > pivot) { increment( hf_right, hf, i ); }
    else if (hf->values[i] == pivot) { increment( hf_pivot, hf, i ); }

  }

/*   for (i=0; i<hf_right->active_size; i++) { */
/*     printf("P: %i VR: %i\n", pivot, hf_right->values[i]); */
/*   } */

  HuffFreq *hf_out = join(qsortHuffFreq( hf_left ), hf_pivot, qsortHuffFreq( hf_right ));

  freeHuff(hf_left);  freeHuff(hf_pivot);  freeHuff(hf_right);
  
  return hf_out ;

}


void increment(HuffFreq *hf_rlp, HuffFreq *hf, long i) {
  hf_rlp->active_size += 1;
  hf_rlp->keys = (char *)realloc(hf_rlp->keys, (sizeof(char) * hf_rlp->active_size));
  hf_rlp->values = (long *)realloc(hf_rlp->values, (sizeof(long) * hf_rlp->active_size));
  hf_rlp->keys[hf_rlp->active_size-1] = hf->keys[i];
  hf_rlp->values[hf_rlp->active_size-1] = hf->values[i];
}

HuffFreq *join( HuffFreq *hf_left, HuffFreq *hf_pivot, HuffFreq *hf_right ) {
  long length = hf_left->active_size + hf_pivot->active_size + hf_right->active_size;
  HuffFreq *hf = hfalloc(length);
  hf->active_size = 0;

  long i=0, j=0;
  for (; i<hf_left->active_size; i++, j++) {
    hf->values[j] = hf_left->values[i];
    hf->keys[j] = hf_left->keys[i];
    hf->active_size += 1;
  }

  for (i=0; i<hf_pivot->active_size; i++, j++) {
    hf->values[j] = hf_pivot->values[i];
    hf->keys[j] = hf_pivot->keys[i];
    hf->active_size += 1;
  }

  for (i=0; i<hf_right->active_size; i++, j++) {
    hf->values[j] = hf_right->values[i];
    hf->keys[j] = hf_right->keys[i];
    hf->active_size += 1;
  } 

  return hf;

}


HuffFreq *hfalloc(long i) {
  HuffFreq *hf = (HuffFreq *)malloc(sizeof(HuffFreq));
  hf->keys = (char *)malloc(sizeof(char) * i);
  hf->values = (long *)malloc(sizeof(long) * i);

  return hf;
}


void freeHuff( HuffFreq *hf) {
  free(hf->keys);  free(hf->values);  free(hf);
}


HuffTreeNode *buildHuffTree( HuffFreq *hf ){
  HuffTreeNode *hfnodes = (HuffTreeNode *)malloc(sizeof(HuffTreeNode) * hf->active_size);

  long i=0;
  for (; i<hf->active_size; i++) {
    HuffTreeNode hn;
    hn.right = 0;
    hn.left = 0;
    hn.symbol = hf->keys[i];
    hn.freq   = hf->values[i];
    hfnodes[i] = hn;
  }

  HuffTreeNode *hftn = mergeHuffTrees(hfnodes, hf->active_size); 

  return hftn;
}


HuffTreeNode *mergeHuffTrees(HuffTreeNode *hfnodes, long length) {
  HuffTreeNode *merged_node = (HuffTreeNode *)malloc(sizeof(HuffTreeNode));;
  HuffTreeNode *small1 = (HuffTreeNode *)malloc(sizeof(HuffTreeNode));
  HuffTreeNode *small2 = (HuffTreeNode *)malloc(sizeof(HuffTreeNode));
  long ind1, ind2;

  if (length == 1){ return &(hfnodes[0]); }

  /* find the trees with the two smallest frequencies */
  long i=0;
  for (; i<length; i++) {
    if (i == 0){ 
      *small1 = hfnodes[i];
      ind1 = i; 
    }
    else if (i == 1) {
      if (hfnodes[i].freq < small1->freq) { 
	*small2 = *small1; ind2 = ind1;
	*small1 = hfnodes[i]; ind1 = 1;
      }
      else {
	*small2 = hfnodes[i]; ind2 = 1;
      }
    }
    else {
      if (hfnodes[i].freq < small1->freq){ *small1 = hfnodes[i]; ind1 = i; }
      else if (hfnodes[i].freq < small2->freq){ *small2 = hfnodes[i]; ind2 = i; }
      else { continue; }
    }
  }

  merged_node->left = small1;
  merged_node->right = small2;
  merged_node->freq = small1->freq + small2->freq;

  /* copy the old array of trees into a new, shorter list */
  HuffTreeNode *new_list = (HuffTreeNode *)malloc(sizeof(HuffTreeNode) * (length - 1));
  long j=0;
  for (i=0; i<length; i++) {
    if (j == 0) {
      new_list[j] = *merged_node; j++;
    }
    else if (i == ind1 || i == ind2) {
      continue;
    }
    else {
      new_list[j] = hfnodes[i]; j++;
    }
  }


  return( mergeHuffTrees(new_list, length - 1) );

}



HuffFreq *buildHuffCodeTable( HuffFreq *hf, HuffTreeNode *htn ) {

  hf->codes = (char *)malloc(sizeof(char) * hf->active_size);
  hf->num_bits_used = (int *)malloc(sizeof(int) * hf->active_size);
  
  unsigned char code =~ 1; /* set to zero */
  int iteration = 0;       /* level in the tree */

  /* tabulateCodes( hf, htn, iteration, code ); */

  return hf;
}

void tabulateCodes( HuffFreq *hf, HuffTreeNode *htn, int iteration, unsigned char code ) {
  unsigned char lcode = code; /* code to send left */
  unsigned char rcode = code; /* code to send right */

  /* left side */
  if ( htn->left != NULL && iteration < 8) {
    lcode <<= 1; /* add a zero in the low bit */
    HuffTreeNode *left = (HuffTreeNode *)htn->left;
    tabulateCodes( hf, left, iteration + 1, lcode);
  }

  /* right side */
  if ( htn->right != NULL && iteration < 8) {
    unsigned char mask =~ 0;  /* all ones */
    mask >>= 7;               /* pad all except low bit with zeros */
    rcode <<= 1;              /* move the bits over 1, pulling a zero into the low bit */
    rcode |= mask;            /* set that first bit on */

    HuffTreeNode *right =  (HuffTreeNode *)htn->right;
    tabulateCodes( hf, right, iteration + 1, rcode);
  }

  hf->codes[(int)htn->symbol] = code;
  hf->num_bits_used[(int)htn->symbol] = iteration;

}


/* generateCompressedArray
 *
 * Basically the idea here is to get rid of all the unused lead zeros that
 * are part of the codes so that we can stack arbitrary number (including
 * fractional parts of the char codes in each char in the compressed array. 
 */
unsigned char *generateCompressedArray( HuffFreq *hf, unsigned char *data, long length){
  unsigned char *output = (char *)malloc(sizeof(char) * length);  /* won't be bigger than this! */
  long olength = 0;
  


  long i=0;
  int more_char_next = FALSE;
  for (; i<length;) {
/*     unsigned char char_one, char_next; */
/*     int space_remaining = 8; */

/*     printf("%i\n", hf->codes[data[i]]); i++; */

/*     /\* grab the first char worth of data or use the remaineder from last loop *\/ */
/*     if (more_char_next == FALSE) { */
/*       i++; */
/*       char_one = hf->codes[data[i]]; */
/*     } */
/*     else { */
/*       char_one = char_next; */
/*       char_next = 0x00; */
/*     } */
    
/*     printf("%i\t%i\n", space_remaining, hf->num_bits_used[data[i]]); */
/*     /\* shift it so the used bits are in the high end of the char *\/ */
/*     space_remaining =- hf->num_bits_used[data[i]]; */
/*     char_one <<= (space_remaining); */

/*     /\* try to load as much data into the remaining bits as possible *\/ */
/*     while (space_remaining > 0){ */
      
/*       i++; */
/*       char_next = hf->codes[data[i]]; */
      
/*       /\* might need to shift left or right *\/ */
/*       if (hf->num_bits_used[data[i]] < space_remaining) { */
/* 	space_remaining =- hf->num_bits_used[data[i]]; */
/* 	char_next <<= space_remaining; */
/* 	char_one &= char_next; */
/* 	more_char_next = FALSE; */
/*       } */
/*       else { */
/* 	char_next >>= (hf->num_bits_used[data[i]] - space_remaining); */
/* 	char_one &= char_next; */
	
/* 	char_next = hf->codes[data[i]];  /\* now put the unrecorded bits back in char_next *\/ */
/* 	char_next &= 0x0f;               /\* mask out the bits already recorded *\/ */
/* 	more_char_next = TRUE; */
/* 	olength += 1; */
/*       } */
/*     } */

/*     output[olength] = char_one; */

  }

/*   /\* chop the array *\/ */
/*   output = (char *)realloc(output, sizeof(char) * olength); */

/*   printf("%i %i\n", length, olength); */

  return output;
}


/* used for debugging */
void viewHuffTree( HuffTreeNode *hfn ) {

  if (hfn->left != NULL) {
    /* printf("going left\n"); */
    viewHuffTree( (HuffTreeNode *)hfn->left );
  }

  if (hfn->right != NULL) {
    /* printf("going right\n"); */
    viewHuffTree( (HuffTreeNode *)hfn->right );
  }

  if (hfn->left == NULL && hfn->right == NULL) {
    printf("leaf: %c %i\n", hfn->symbol, hfn->freq);
  }

}


void viewHuffFreq(HuffFreq *hf) {
  long i=0;
  for (; i<hf->active_size; i++){
    printf("%i: %c\t%c\t%i\n", i, hf->keys[i], hf->codes[i], hf->num_bits_used[i]);
  }

}
