/* TO DO'S
 *
 *
 * copyright University of Victoria, 2004
 */


#include "logo.h"
#include "globals.h"
#include "sound.h"
#include <stdlib.h>
#include <float.h>

int location = 0; /* index of current sample */

NODE *lsinewave(NODE *args) {
  NODE *f, *array;
  int frequency;

  f = numeric_arg(args);
  if (NOT_THROWING) {
    frequency = (nodetype(f) == FLOATT) ? (FIXNUM)getfloat(f) : getint(f);
  }

  int table_size = SAMPLE_RATE/frequency;
  array = make_array(table_size);
  setarrorg(array, 0);

  int i;
  for (i=0; i<getarrdim(array); i++) {
    float sample = (float) (AMPLITUDE * sin( (((double)i/(double)(table_size)) * 2 * M_PI )));
    (getarrptr(array))[i] = newnode(FLOATT);
    setfloat((getarrptr(array))[i], sample);
  }

  return(array);
}


NODE *ltrianglewave(NODE *args) {
  NODE *f, *array;
  int frequency;

  f = numeric_arg(args);
  if (NOT_THROWING) {
    frequency = (nodetype(f) == FLOATT) ? (FIXNUM)getfloat(f) : getint(f);
  }

  int table_size = SAMPLE_RATE/frequency;
  array = make_array(table_size);
  setarrorg(array, 0);

  double x = 0.0;
  double deltax = (2 * M_PI) / (double)table_size;

  int i;
  for (i=0; i<getarrdim(array); i++) {
    float sample = 0.0;

    /* The pos variable limits calcultion to one triangle which can then be duplicated
     * and inverted to complete the cycle. */
    double pos = x;
    if (pos > M_PI){ pos = x - M_PI; }

    if (pos <= M_PI/2){
      sample = (0.9/(M_PI/2)) * pos;
    }
    else {
      sample = (((-0.9/(M_PI/2))) * pos) + 1.8;
    }
    
    if (x > M_PI && x <= 2*M_PI) { sample*=-1; }


    (getarrptr(array))[i] = newnode(FLOATT);
    setfloat((getarrptr(array))[i], sample);

    x += deltax;
  }

  return(array);
}


NODE *lsquarewave(NODE *args) {
  NODE *f, *array;
  int frequency;

  f = numeric_arg(args);
  if (NOT_THROWING) {
    frequency = (nodetype(f) == FLOATT) ? (FIXNUM)getfloat(f) : getint(f);
  }

  int table_size = SAMPLE_RATE/frequency;
  array = make_array(table_size);
  setarrorg(array, 0);

  double x = 0.0;
  double deltax = (2 * M_PI) / (double)table_size;

  int i;
  float sample;
  for (i=0; i<getarrdim(array); i++) {
    if (x <= M_PI){ sample = 0.9; }
    else { sample = -0.9; }

    (getarrptr(array))[i] = newnode(FLOATT);
    setfloat((getarrptr(array))[i], sample);
    
    x += deltax;
  }  
  

  return(array);
}


NODE *lnoise(NODE *args) {
  NODE *oarray, *d;
  long duration;  /* number of milliseconds of noise to return */

  d = numeric_arg(args);
  if (NOT_THROWING){ duration = (nodetype(d) == FLOATT? (FIXNUM)getfloat(d): getint(d)); }
  else { err_logo(BAD_DATA, d); return UNBOUND; }

  long length = duration * SAMPLE_RATE/1000;
  oarray = make_array(length);
  setarrorg(oarray, 0);

  int i=0, r;
  float sample;
  for (; i<length; i++){
    r = rand(); 
    sample = ((float)r/(RAND_MAX/2)) - 1.0;
 
    (getarrptr(oarray))[i] = newnode(FLOATT);
    setfloat((getarrptr(oarray))[i], sample);   
  }

  return(oarray);
}

NODE *lcopywave(NODE *args) {
  NODE *new_array, *array, *times;
  int i, j, k, n, size;

  array = car(args);
  times = numeric_arg(cdr(args));

  if (NOT_THROWING) {
    n = (nodetype(times) == FLOATT) ? (FIXNUM)getfloat(times) : getint(times);
  }

  if (nodetype(array) != ARRAY) {
    err_logo(BAD_DATA, array);
  }
  else if (n < 0) {
    err_logo(BAD_DATA, times);
  }

  size = getarrdim(array) * n;
  new_array = make_array(size);
  setarrorg(new_array, 0);

  /* works, but too slowly... i can't think of any method for speeding it up
   * such as an array of pointers, some struct that maintains one copy of the
   * the cycle plus the number of iterations, etc... that doesn't necessitate
   * chaning the underlying logo data structure logo_node.  but, maybe i'm
   * missing something? */
  k=0;
  for (i=0; i<n; i++) {
    for (j=0; j<getarrdim(array); j++){
      (getarrptr(new_array))[k] = newnode(FLOATT);
      setfloat((getarrptr(new_array))[k], getfloat((getarrptr(array))[j]));
      k++;
    }
  }

  return(new_array);
}


NODE *lwaveenvelope(NODE *args) {
  NODE *array, *liSt, *pair, *a, *d;
  int i;

  array = car(args);
  liSt = (NODE *)list_arg(cdr(args));

  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }
  if (!is_list(liSt)){ err_logo(BAD_DATA, liSt); return UNBOUND; }

  float factor = 1.0;

  float currentmsec, sample;
  float oldx = 0.0;
  float oldy = 0.0;
  float thresholdmsec = 0.0;
  float thresholdfactor = 0.0;
  for (i=0; i<getarrdim(array); i++) {
    sample = getfloat((getarrptr(array))[i]);
    currentmsec = (float)i/((float)SAMPLE_RATE/1000.0);

    if (currentmsec >= thresholdmsec) { /* we have crossed a time threshold */
 
      oldx = thresholdmsec;
      oldy = thresholdfactor;

      if (NOT_THROWING && liSt != NIL) { /* check args and retrieve their values */
	pair = car(liSt);
	liSt = cdr(liSt);

	if (is_list(pair) && NOT_THROWING) {
	  a = numeric_arg(pair);
	  d = numeric_arg(cdr(pair));

	  if (NOT_THROWING) {
	    thresholdfactor = (nodetype(a) == FLOATT? getfloat(a): (FLONUM)getint(a));
	    thresholdmsec += (nodetype(d) == FLOATT? (FIXNUM)getfloat(d): getint(d));

	    while (thresholdfactor > 1) { thresholdfactor = thresholdfactor / 10; }
	  }
	  else { err_logo(BAD_DATA, pair); return UNBOUND; }
	}
	else { err_logo(BAD_DATA, pair); return UNBOUND; }
      }
    }

    float rise = thresholdfactor - oldy;
    float run  = thresholdmsec - oldx;
    float intersection = oldy - ((rise/run)*oldx);

    factor = (rise/run)*currentmsec + intersection;
    sample *= factor;
    setfloat((getarrptr(array))[i], sample);
  }

  return(array);
}


NODE *lcombinewaves(NODE *args) {
  NODE *liSt, *farray, *carray;
  int longest = 0, n = 0;

  liSt = car(args);
  if (!is_list(liSt) || !NOT_THROWING){ err_logo(BAD_DATA, args); return UNBOUND; }

  /* determine which array is longest */
  while (liSt != NIL && NOT_THROWING) {
    carray = car(liSt); liSt = cdr(liSt);

    if (nodetype(carray) != ARRAY){ err_logo(BAD_DATA, carray); return UNBOUND; }
    if (nodetype((getarrptr(carray))[0]) != FLOATT){ err_logo(BAD_DATA, carray); return UNBOUND; }

    if (longest < getarrdim(carray)){ longest = getarrdim(carray); }
    n++;
  }

  /* create the new array which has the length of the longest argument array */
  farray = make_array(longest);
  setarrorg(farray, 0);

  /* now copy the arrays in such that they fill the whole new array...
   * note that this could mean some waves appear to end very abruptly, ie
   * not at a near-zero value.  one could take care of this using the 
   * waveenvelope function. */
  liSt = car(args);
  int i=0;
  for (; i<longest; i++){
    float sample=0;

    while (liSt != NIL){
      carray = car(liSt); liSt = cdr(liSt);
 
      int j = i;
      while (j >= getarrdim(carray)){ j = j - getarrdim(carray);}

      sample += getfloat((getarrptr(carray))[j]);
    }
    liSt = car(args); /* resest liSt */

    sample = sample/n;
    (getarrptr(farray))[i] = newnode(FLOATT);
    setfloat((getarrptr(farray))[i], sample);
  }


  return(farray);
}


NODE *laddwaveat(NODE *args) {
  NODE *o, *farray, *tarray, *array;
  unsigned long offset;

  tarray = car(args); args = cdr(args);    /* target array */
  array  = car(args); args = cdr(args);    /* array to add */
  if (nodetype(tarray) != ARRAY){ err_logo(BAD_DATA, tarray); return UNBOUND; }
  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }


  o = numeric_arg(args);
  if (NOT_THROWING){
    offset = (nodetype(o) == FLOATT? (FIXNUM)getfloat(o): getint(o));
  }
  else { err_logo(BAD_DATA, o); return UNBOUND; }

  offset *= 44.1;  /* convert offset from msec to samples */
  
  /* create the final array */
  unsigned long length = 0;
  if (getarrdim(tarray) > offset + getarrdim(array)){ length = getarrdim(tarray); }
  else { length = offset + getarrdim(array); }
  farray = make_array(length);
  setarrorg(farray, 0);

  unsigned long i=0,j=0;
  for (; i<getarrdim(farray); i++){
    float sample = 0.0;

    if (i < getarrdim(tarray)){ sample += getfloat((getarrptr(tarray))[i]); }
    if (i >= offset && j < getarrdim(array)){ sample += getfloat((getarrptr(array))[j]); j++;}

    sample = sample / 2;  /* for now, we're going to normalize automatically */

    (getarrptr(farray))[i] = newnode(FLOATT);
    setfloat((getarrptr(farray))[i], sample);
  
  }
  

  return(farray);
}


NODE *lhighpassfilter(NODE *args) {
  return( filter(args) );
}


NODE *llowpassfilter(NODE *args) {
  return( filter(args) );
}


NODE *lplaywave(NODE *args) {
  PaError err;
  PortAudioStream *stream;

  if (SOUNDON != 1) { return(UNBOUND); }                /* ensure portaudio has been initialized */
  if (nodetype(car(args)) != ARRAY ||
      nodetype((getarrptr(car(args)))[0]) != FLOATT) {  /* check the arguments right away */
    err_logo(BAD_DATA, args);
    return(UNBOUND);
  }

  err = Pa_OpenStream(
		      &stream,
		      paNoDevice,     /* default input device */
		      0,              /* no input */
		      paFloat32,      /* 32 bit floating point input */
		      NULL,
		      Pa_GetDefaultOutputDeviceID(),
		      2,              /* stereo output */
		      paFloat32,      /* 32 bit floating point output */
		      NULL,
		      SAMPLE_RATE,
		      FRAMES_PER_BUFFER,
		      0,              /* number of buffers, if zero then use default minimum */
		      paClipOff,      /* clipping accounted for elsewhere */
		      playwaveCallback,
		      args );
  if (err != paNoError){ error(); }
  err = Pa_StartStream( stream );
  if( err != paNoError ) { error(); }
  
  while (location < getarrdim(car(args))) { /* poll until done */
    Pa_Sleep( 10 );
  }
  
  err = Pa_StopStream( stream );
  if( err != paNoError ) { error(); }
  err = Pa_CloseStream( stream );
  if( err != paNoError ) { error(); }
  
  /* reset globals */
  location = 0;

  return(UNBOUND);
}

NODE *lrecordwave(NODE *args){ 
  PaError err;
  PortAudioStream *stream;
  NODE *oarray, *n;
  int time;

  if (SOUNDON != 1) { return(UNBOUND); } /* ensure portaudio has been initialized */

  n = numeric_arg(args);

  if (NOT_THROWING) {
    time = (nodetype(n) == FLOATT) ? (FIXNUM)getfloat(n) : getint(n);
  }
  else { err_logo(BAD_DATA, n); return UNBOUND; }

  if (time < 0) { err_logo(BAD_DATA, n); }

  oarray = make_array( time * SAMPLE_RATE/1000 );  /* will hold the recorded floats*/
  setarrorg(oarray, 0);

  /* this is a hack which I haven't yet been able to remove. */
  /* for some reason i need to set these nodes before passing this */
  /* array to the callback below.  must be something simple, but I */
  /* don't see it!!! arghhhh.... */
  unsigned long i = 0;
  for (; i<getarrdim(oarray); i++) {
    getarrptr(oarray)[i] = newnode( FLOATT );
  }


/*     err = Pa_OpenStream( */
/*               &stream, */
/*               Pa_GetDefaultInputDeviceID(), */
/*               NUM_CHANNELS, */
/*               PA_SAMPLE_TYPE, */
/*               NULL, */
/*               paNoDevice, */
/*               0, */
/*               PA_SAMPLE_TYPE, */
/*               NULL, */
/*               SAMPLE_RATE, */
/*               FRAMES_PER_BUFFER,            /\* frames per buffer *\/ */
/*               0,               /\* number of buffers, if zero then use default minimum *\/ */
/*               0, /\* paDitherOff, // flags *\/ */
/*               recordCallback, */
/*               &data ); */

  err = Pa_OpenStream(
		      &stream,
		      Pa_GetDefaultInputDeviceID(),     /* default input device */
		      1,
		      paFloat32,      /* 32 bit floating point input */
		      NULL,
		      paNoDevice,
		      0,              /* stereo output */
		      paFloat32,      /* 32 bit floating point output */
		      NULL,
		      SAMPLE_RATE,
		      FRAMES_PER_BUFFER,
		      0,              /* number of buffers, if zero then use default minimum */
		      paClipOff,      /* clipping accounted for elsewhere */
		      recordwaveCallback,
		      oarray );

  location = 0; /* just in case it has't been reset somewhere */

  if (err != paNoError){ error(); }
  err = Pa_StartStream( stream );
  if( err != paNoError ) { error(); }

  printf("start stream\n");

  Pa_Sleep( time );
/*   while (location < getarrdim(oarray)) { */
/*     Pa_Sleep( 10 ); */
/*   } */

  printf("stop stream\n");

  err = Pa_StopStream( stream );
  if( err != paNoError ) { error(); }
  err = Pa_CloseStream( stream );
  if( err != paNoError ) { error(); }

  location = 0; /* reset this global */

  return oarray; 
}


NODE *lcutwave(NODE *args){ 
  NODE *iarray, *oarray; /* input array, output array */
  NODE *liSt;            /* two item segment list [start stop] */
  NODE *b, *e;
  int begin, end, length;

  iarray = car(args);
  if (nodetype(iarray) != ARRAY){ err_logo(BAD_DATA, iarray); return UNBOUND; }

  liSt = (NODE *)list_arg(cdr(args));
  if (!is_list(liSt)){ err_logo(BAD_DATA, liSt); return UNBOUND; }

  b = numeric_arg(liSt); liSt = cdr(liSt);
  if (NOT_THROWING){ begin = (nodetype(b) == FLOATT? (FIXNUM)getfloat(b): getint(b)); }
  else { err_logo(BAD_DATA, b); return UNBOUND; }

  e = numeric_arg(liSt);
  if (NOT_THROWING){ end = (nodetype(e) == FLOATT? (FIXNUM)getfloat(e): getint(e)); }
  else { err_logo(BAD_DATA, e); return UNBOUND; }

  length = end - begin;
  if (length < 0){ length *= -1; }

  oarray = make_array(length);
  setarrorg(oarray, 0);
  
  int i=0;
  for (; i<length; i++){
    float sample = 0.0;

    if (i<getarrdim(iarray)){
      sample = getfloat((getarrptr(iarray))[i]);
    }

    (getarrptr(oarray))[i] = newnode(FLOATT);
    setfloat((getarrptr(oarray))[i], sample); 
  }

  return oarray; 
}


NODE *lwriteaudio(NODE *args){ 
  NODE *filenamenode, *array;

  /* using procedures from files.c */
  filenamenode = car(args);  args = cdr(args);
  lopen(filenamenode, "w");

  array = car(args);
  if (nodetype(array) != ARRAY){ err_logo(BAD_DATA, array); return UNBOUND; }

  unsigned long i=0, j=0;
  for (; i<getarrdim(array); i++){
    float sample = 0.0;
    sample = getfloat((getarrptr(array))[i]);

    float buffer[1024];
    if (j < 1024 && i != (getarrdim(array) -1)){ buffer[j] = sample; j+=1; }
    else {
      if (fwrite(buffer, sizeof(float), j, (FILE *)file_list->n_obj) < j){
	err_logo(BAD_DATA_UNREC, car(args));
      }

      j = 0; /* reset to fill next buffer */
    }

  }
  

  lclose(filenamenode);

  return UNBOUND; 
}


NODE *lreadaudio(NODE *args){ 
  NODE *filenamenode, *array;

  /* using procedures from files.c */
  filenamenode = car(args);  args = cdr(args);

  long length = openforread(filenamenode);
  length = length/sizeof(float);
  array = make_array(length);

  /* load the array of FLOATTs */
  float buffer[1024];
  long i=0;
  while (!feof((FILE *)file_list->n_obj) && !ferror((FILE *)file_list->n_obj)){
    int num = fread(buffer, sizeof(float), 1024, (FILE *)file_list->n_obj);

    long j=0;
    for (; j<1024; j++){

      if (i < length){
	(getarrptr(array))[i] = newnode(FLOATT);
	setfloat((getarrptr(array))[i], buffer[j]);
      }
      i+=1;

    }

  }
  

  lclose(filenamenode);

  return array; 
}




long openforread(NODE *filenamenode){
  FILE *tmp;


  /* using procedures from files.c */

  /* this, unfortunately, is a repeat of the code in lopen, but it was necessary to
     remove a level of indirection to get at filehand directly and test if the file
     already exists. */
  if (find_file(filenamenode, FALSE) != NULL){
    err_logo(FILE_ERROR, make_static_strnode(message_texts[ALREADY_OPEN]));
    return(-1);
  }
  else if ((tmp = open_file(filenamenode, "r")) != NULL) {
    push(filenamenode, file_list);
    file_list->n_obj = (NODE *)tmp;
  }
  else {
    err_logo(FILE_ERROR, make_static_strnode(message_texts[CANT_OPEN]));
    return(-1);
  }

  /* create the appropriate size array */
  fseek((FILE *)file_list->n_obj, 0, SEEK_END);
  long length = ftell((FILE *)file_list->n_obj);
  fseek((FILE *)file_list->n_obj, 0, SEEK_SET);

  return length;

}

int recordwaveCallback( void *inputBuffer, void *outputBuffer,
			unsigned long framesPerBuffer,
			PaTimestamp outTime, void *userData ) {


  float *input = (float *)inputBuffer;
  NODE *buffer = (NODE *)userData;
  unsigned long i = 0;

  unsigned long framesLeft;

  if (getarrdim(buffer) - location < framesPerBuffer) {
    framesLeft = getarrdim(buffer) - location;
  }
  else {
    framesLeft = framesPerBuffer;
  }

  for (; i<framesLeft; i++) {
    setfloat(getarrptr(buffer)[location], *input++);
    location += 1;
  }


  return 0;
}

int playwaveCallback( void *inputBuffer, void *outputBuffer,
		      unsigned long framesPerBuffer,
		      PaTimestamp outTime, void *userData ) {
  float *out = (float *)outputBuffer;
  NODE *args = (NODE *)userData;
  NODE *array = car(args); /* type should already have been checked */
  unsigned long i = 0;
  
  for (; i<framesPerBuffer; i++) {
    float sample;
    if (location < getarrdim(array)){ 
      /* this is necessary in case we have some empty indices */
      if (nodetype((getarrptr(car(args)))[location]) != FLOATT) {
	sample = 0.0;
      }
      else {
	sample = (float)getfloat((getarrptr(array))[location]); 
      }
    }
    else { sample = 0.0; }

    *out++ = sample; /* left */
    *out++ = sample; /* right */

    location++;
  }

  
  return( 0 );
}

/* Yes this looks like a logo primitive function.... It's not but is instead wrapped in the
 * lowpassfilter and highpassfilter functions */
NODE *filter(NODE *args){
  NODE *iarray, *oarray, *f;   /* input array, output array and cutoff frequency respectively */
  int freq;
  int numcoeffs = 3;
  float bucket[3];
  float coeffs[3];

  
  iarray = car(args);
  if (nodetype(iarray) != ARRAY){ err_logo(BAD_DATA, oarray); return UNBOUND; }

  f = numeric_arg(cdr(args));
  if (NOT_THROWING){ freq = (nodetype(f) == FLOATT? (FIXNUM)getfloat(f): getint(f)); }
  else { err_logo(BAD_DATA, f); return UNBOUND; }

  oarray = make_array(getarrdim(iarray));
  setarrorg(oarray, 0);
  
  set_filter_coefficients(coeffs);
  initialize_fir_bucket(bucket, iarray, 3);
 
  int i=0, now=1;
  for (; i<getarrdim(iarray); i++){
    float sample = coeffs[0];

    int j=1, k=now;
    for (; j<3; j++) {
      sample += (bucket[k] * coeffs[j]);
      if (k >= 2){ k = 0; } else { k += 1; }
    }

    (getarrptr(oarray))[i] = newnode(FLOATT);
    setfloat((getarrptr(oarray))[i], sample);

    if (now >= 2){ now = 0; } else { now += 1; }

    if (i+2 < getarrdim(iarray)){
      bucket[now] = getfloat((getarrptr(iarray))[i+2]);
    }
    else { /* should do something more clever here */
      bucket[now] = 0.0;
    }

  }

  return(oarray);
}


void set_filter_coefficients(float *coeffs){
  int rows = SIMPLEX_ROWS, cols = SIMPLEX_COLS;
  int most_neg_col, best_ratio_row;

  /* This is some test data.  The first three rows are data from pure sine waves below, at and
   * above the cutoff-- 220, 440, 660 with cutoff around 440.  The next row is the amplitude
   * contraint the a1 + a2 + a3 <= 1.8.  Where does 1.8 come from?  Our amplitude ranges from
   * -0.9 to 0.9, the zero has been shifted to remove the negatives so as not to confuse simplex
   * (perhaps this isn't really necessary... should check).  The final row is the row we seek to
   * maximize.  It comes from real data, ie. the real wave we wish to filter. */
  float matrix[5][9] = { {1.0, 0.140791013836861, 0.168643176555634, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0},
			 {1.0, 0.278115302324295, 0.331312090158463, 0.0, 1.0, 0.0, 0.0, 0.0},
			 {1.0, 0.412403881549835, 0.486576735973358, 0.0, 0.0, 1.0, 0.0, 0.0, 0.334496200084686},
			 {1.0, 1.0, 1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.9},
			 {-1.0, -0.276597440242767, -0.327609956264496, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0}
  };

/*   float matrix[4][9] = { {1.0, 0.140791013836861, 0.168643176555634, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0}, */
/* 			 {1.0, 0.412403881549835, 0.486576735973358, 0.0, 0.0, 1.0, 0.0, 0.0, 0.334496200084686}, */
/* 			 {1.0, 1.0, 1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.9}, */
/* 			 {-1.0, -0.276597440242767, -0.327609956264496, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0} */
/*   }; */

/*   float matrix[4][7] = { */
/*     {1.0, 3.0, 1.0, 1.0, 0.0, 0.0, 45.0}, */
/*     {2.0, 3.0, 3.0, 0.0, 1.0, 0.0, 50.0}, */
/*     {4.0, 2.0, 1.0, 0.0, 0.0, 1.0, 60.0}, */
/*     {-6.0, -3.0, -2.0, 0.0, 0.0, 0.0, 0.0} */
/*   }; */

/*   float matrix[3][5] = { */
/*     {2.0, 1.0, 1.0, 0.0, 104.0}, */
/*     {1.0, 2.0, 0.0, 1.0, 76.0}, */
/*     {-6.0, -11.0, 0.0, 0.0, 0.0} */
/*   }; */

  while (still_contains_negatives(matrix, cols) != -1){
    most_neg_col = find_most_negative_column(matrix[SIMPLEX_ROWS - 1], cols);
    best_ratio_row = find_best_ratio_row( matrix, cols, rows, most_neg_col );
    adjust_pivot_row(matrix, best_ratio_row, most_neg_col, cols);
    zero_pivot_column_of_other_rows(matrix, best_ratio_row, most_neg_col, rows, cols);
  
    print_matrix(matrix, rows, cols);
    printf("===============================================================================\n");
  }

  /* for dev only */
  float temp[3] = {-.897, -0.805, 0.2888};
  coeffs = temp;

}

int still_contains_negatives(float matrix[][SIMPLEX_COLS], int cols){

  int i=0;
  for (; i<cols; i++){
    if (matrix[SIMPLEX_ROWS-1][i] < 0){ return(1); }
  }

  return(-1);
}

void zero_pivot_column_of_other_rows(float matrix[][SIMPLEX_COLS], 
				     int best_ratio_row, int most_neg_col, 
				     int rows, int cols){

  int i=0; /* as in th row */
  for (; i<rows; i++){
    if (i == best_ratio_row){ continue; }

    float scaling = matrix[i][most_neg_col]/matrix[best_ratio_row][most_neg_col];

    int j=0; /* as in jth column */
    for (; j<cols; j++){
      matrix[i][j] = matrix[i][j] - (matrix[best_ratio_row][j]*scaling);
    }
  }

}

void adjust_pivot_row(float matrix[][SIMPLEX_COLS],  int best_ratio_row,  int most_neg_col, int cols){
  float pivot = matrix[best_ratio_row][most_neg_col];

  int i=0;
  for (; i<cols; i++){
    matrix[best_ratio_row][i] = matrix[best_ratio_row][i]/pivot;
  }
  

}

void print_matrix(float matrix[][SIMPLEX_COLS], int rows, int cols){
  int i=0, j;
  
  for (; i<rows; i++){
    for (j=0; j<cols; j++){
      printf("%f  ", matrix[i][j]);
    }
    printf("\n");
  }

}

int find_most_negative_column(float *array, int n) {
  float lowest = FLT_MAX;  
  int c = -1;

  int i=0;
  for (; i<n; i++){
    if (array[i] < lowest){ lowest = array[i]; c=i; }
  } 

  return(c);
}

int find_best_ratio_row( float matrix[][SIMPLEX_COLS], int cols, int rows, int most_neg_col ){
  float best_ratio = FLT_MAX;
  int c = -1;
  
  int i=0;
  for (; i<rows-1; i++){

    if (matrix[i][cols-1] >= 0.0 && best_ratio > matrix[i][cols-1]/matrix[i][most_neg_col]) {
      best_ratio = matrix[i][cols-1]/matrix[i][most_neg_col];
      c = i;
    }
  }

  return(c);
}


void initialize_fir_bucket(float *bucket, NODE *array, int n){  

  /* The bucket is loaded such that the index at 0 is the latest sample. */
  int i=n-1;
  for (; i>=0; i--){
    bucket[i] = (float)getfloat((getarrptr(array))[i]);
  }
}


/* NODE *lcompressfile( NODE *args ){ */
/*   NODE *filenamenode; */
/*   unsigned char *data, *compresseddata; */
/*   HuffFreq *hf; */

/*   filenamenode = car(args); */
/*   long length=openforread(filenamenode); */

/*   data = (char *)malloc(sizeof(char) * length); */
/*   unsigned char buffer[1024]; */
/*   long i=0, j; */
/*   while (!feof((FILE *)file_list->n_obj) && */
/* 	 !ferror((FILE *)file_list->n_obj) && */
/* 	 i < length) { */
/*     int num = fread(buffer, sizeof(char), 1024, (FILE *)file_list->n_obj); */
/*     for (j=0; j<num; j++){ data[i] = buffer[j]; i+=1;} */
    
/*   } */

/*   hf = getFreqCounts(data, length); */
/*   /\* actually the sort is unnecessary, but this call still is...  */
/*    * it's used to strip out keys with frequencies of zero... */
/*    * needs to be rewritten just to do that! *\/ */
/*   hf = sortHuffFreq( hf );   */
/*   HuffTreeNode *hftree = buildHuffTree( hf ); */
/*   hf = buildHuffCodeTable( hf, hftree ); */
/*   compresseddata = generateCompressedArray(hf, data, length); */
  

/*   HuffTreeNode *hf1 = (HuffTreeNode *)malloc(sizeof(HuffTreeNode)); */
/*   HuffTreeNode *hf2 = (HuffTreeNode *)malloc(sizeof(HuffTreeNode)); */
/*   HuffTreeNode *hf3 = (HuffTreeNode *)malloc(sizeof(HuffTreeNode)); */

/*   hf3->left = 0; */
/*   hf3->right = 0; */
/*   hf3->symbol = 'a'; */
/*   hf3->freq = 30; */

/*   hf2->left = 0; */
/*   hf2->right = 0; */
/*   hf2->symbol = 'b'; */
/*   hf2->freq = 20; */

/*   hf1->left = 0; */
/*   hf1->right = 0; */
/*   hf1->symbol = 'c'; */
/*   hf1->freq = 15; */

/*   HuffTreeNode hftests[1] = {hf1}; */
/*   viewHuffTree( mergeHuffTrees(hftests, 1) ); */

/*   HuffTreeNode hftests[2] = {hf1, hf2}; */
/*   viewHuffTree( mergeHuffTrees(hftests, 2) ); */

/*   HuffTreeNode hftests[3] = {hf1, hf2, hf3}; */
/*   viewHuffTree( mergeHuffTrees(hftests, 3) ); */

  /* viewHuffTree( hftree ); */
  /* viewHuffFreq( hf ); */

/*   printf("length %i\n", hf->active_size); */

/*   for (i=0; i<hf->active_size; i++) { */
/*     printf("%i\t%i\n", (int)hf->keys[i], hf->values[i]); */
/*   } */
  

/*   free(data); */
/*   lclose(filenamenode); */

/*   return UNBOUND; */
/* } */


NODE *lfft( NODE *args ) {

  return( UNBOUND );
}
